//
// Copyright (C) 2006 Henning Westerholt
// Copyright (C) 2003 CTIE, Monash University
// Copyright (C) 2000 Institut fuer Telematik, Universitaet Karlsruhe
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#include "EtherStreamApp.h"
#include "EtherApp_m.h"

Define_Module(EtherStreamApp);

void EtherStreamApp::initialize(const int stage)
{
     // we can only initialize in the 2nd stage (stage 1), because
    // assignment of "auto" MAC addresses takes place in stage 0
    if (stage!=1) return;

    msgLength = &par("msgLength");
    waitTime = &par("waitTime");

    localSAP = ETHERAPP_CLI_SAP;
    remoteSAP = ETHERAPP_SRV_SAP;

    // statistics
    packetsSent = packetsReceived = seqNum = 0;
    WATCH(packetsSent);
    WATCH(packetsReceived);
    WATCH(seqNum);

    destMACAddress = resolveDestMACAddress();

    // if no dest address given, nothing to do
    if (destMACAddress.isUnspecified())
        return;

    registerDSAP(localSAP);

    cMessage *timermsg = new cMessage("generateNextPacket");
    double d = waitTime->doubleValue();
    scheduleAt(simTime()+d, timermsg);
}

MACAddress EtherStreamApp::resolveDestMACAddress()
{
    MACAddress destMACAddress;
    const char *destAddr = par("destAddress");
    const char *destStation = par("destStation");
    if (strcmp(destAddr,"") && strcmp(destStation,""))
    {
        error("only one of the `destAddress' and `destStation' module parameters should be filled in");
    }
    else if (strcmp(destAddr,""))
    {
        destMACAddress.setAddress(destAddr);
    }
    else if (strcmp(destStation,""))
    {
        std::string destModName = std::string(destStation) + ".mac";
        cModule *destMod = simulation.moduleByPath(destModName.c_str());
        if (!destMod)
            error("module `%s' (MAC submodule of `destStation') not found", destModName.c_str());
        destMACAddress.setAddress(destMod->par("address"));
    }
    return destMACAddress;
}


void EtherStreamApp::handleMessage(cMessage *msg)
{
    if (msg->isSelfMessage())
    {
        sendPacket();
        double d = waitTime->doubleValue();
        // send, then reschedule next sending
        scheduleAt(simTime() + d, msg);
    }
    else
    {
        receivePacket(msg);
    }
}

void EtherStreamApp::registerDSAP(const int dsap)
{
    EV << fullPath() << " registering DSAP " << dsap << "\n";

    Ieee802Ctrl *etherctrl = new Ieee802Ctrl();
    etherctrl->setDsap(dsap);
    cMessage *msg = new cMessage("register_DSAP", IEEE802CTRL_REGISTER_DSAP);
    msg->setControlInfo(etherctrl);

    send(msg, "out");
}

void EtherStreamApp::sendPacket()
{
    seqNum++;

    char msgname[30];
    sprintf(msgname, "req-%d-%ld", id(), seqNum);
    EV << "Generating packet `" << msgname << "'\n";

    EtherAppReq *datapacket = new EtherAppReq(msgname, IEEE802CTRL_DATA);

    datapacket->setRequestId(seqNum);

    long len = msgLength->longValue();
    datapacket->setByteLength(len);

    Ieee802Ctrl *etherctrl = new Ieee802Ctrl();
    etherctrl->setSsap(localSAP);
    etherctrl->setDsap(remoteSAP);
    etherctrl->setDest(destMACAddress);
    datapacket->setControlInfo(etherctrl);

    send(datapacket, "out");
    packetsSent++;
}

void EtherStreamApp::receivePacket(cMessage *msg)
{
    EV << "Received packet `" << msg->name() << "'\n";

    packetsReceived++;
    delete msg;
}

void EtherStreamApp::finish()
{
    if (par("writeScalars").boolValue())
    {
        recordScalar("packets sent", packetsSent);
        recordScalar("packets rcvd", packetsReceived);
    }
}
