/*
 * Copyright (C) 2006 Henning Westerholt
 * Copyright (C) 2003 CTIE, Monash University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifdef _MSC_VER
#pragma warning(disable:4786)
#endif

#include <stdio.h>
#include <stdlib.h>
#include <cxmlelement.h>

#include "MACRelayUnitNPTDMA.h"
#include "EtherFrame_m.h"
#include "utils.h"
#include "Ethernet.h"
#include "MACAddress.h"

using namespace std;

Define_Module( MACRelayUnitNPTDMA );


MACRelayUnitNPTDMA::MACRelayUnitNPTDMA() : timeSlotCount(0), timeSlotLength(0), actualSlot(0), missedSlots(0), isSlotted(0), schedulePresent(false), noRealTimeSlot(0), safetyInterval(0), interFrameGap(0)
{
    schedule = new vector<vector<IOHolder> >();
}

MACRelayUnitNPTDMA::~MACRelayUnitNPTDMA() 
{
    delete schedule;
}

void MACRelayUnitNPTDMA::initialize()
{
    MACRelayUnitNP::initialize();

    timeSlotLength = par("timeSlotLength");

    isSlotted = timeSlotLength > 0;

    if (isSlotted) {
        // open the schedule file
        cXMLElement * scheduleConfig;
        scheduleConfig = par("scheduleConfigFile");
        // get the timeslot count
        timeSlotCount = atoi(scheduleConfig->getAttribute("timeSlotCount"));
        // timeSlotCount starts with 0, and append a slot for no-realtime traffic
        // TODO 64 bytes is not enough for no-rt
        timeSlotCount+=2;
        // the no real-time slot is the last slot in the schedule
        noRealTimeSlot = timeSlotCount -1;

        // construct a parameter resolver
        cXMLElement::ParamResolver * resolver = new ModNameParamResolver(this);
        // get the xml sub tree for this switch, this is where the swith id equals the parent module name
        scheduleConfig = scheduleConfig->getElementByPath("//switch[@id=$PARENTMODULE_FULLNAME]", scheduleConfig, resolver);

        if (scheduleConfig != 0) {
            // cXMLElementList is a typedef for a vector
            cXMLElementList timeSlots = scheduleConfig->getChildren();

            // fill the time slot vector with empty IO holders
            vector<IOHolder> tmp;
            for (unsigned int i = 0; i < timeSlotCount; i++)
                schedule->push_back(tmp);

            // iterate over all timeslots
            cXMLElementList::iterator tIter = timeSlots.begin();

            // copy all time slots in the schedule vector for faster access
            while (tIter != timeSlots.end()) {
                // iterate over all connections
                cXMLElementList connections = (*tIter)->getChildren();
                cXMLElementList::iterator cIter = connections.begin(); 

                // for every connection, append a ioholder object
                while (cIter != connections.end()) {
                    IOHolder io(atoi((*cIter)->getAttribute("in")), atoi((*cIter)->getAttribute("out")));
                    tmp.push_back(io);
                    //cout << "append schedule for slot " << (*tIter)->getAttribute("id") << " with input port " << (*cIter)->getAttribute("in") << " and output port " << (*cIter)->getAttribute("out") << "\n";
                    cIter++;
                }
                // overwrite the empty placeholder
                (*schedule)[atoi((*tIter)->getAttribute("id"))] = tmp;
                // clear the vector for the next round
                tmp.clear();
                tIter++;
            }
            delete resolver;
            schedulePresent = true;
        }
        else 
            EV << "Warning, switch is slotted but has no assigned schedule" << "\n";
    }

    WATCH(timeSlotLength);
    WATCH(isSlotted);
    WATCH(timeSlotCount);
    WATCH(actualSlot);
    WATCH(schedulePresent);
    WATCH(missedSlots);

    EV << "time slot length: " << timeSlotLength << " s" << "\n";
    EV << "time slot count: " << timeSlotCount << "\n";
    EV << "cycle time: " << timeSlotLength * (timeSlotCount -1) << " s" << "\n";
    //EV << "vector" << schedule->size() << endl;
    EV << "\n";

    // schedule first time slot event
    slotTimeEvent = new cMessage("slot");
    scheduleAt(simTime() + timeSlotLength, slotTimeEvent);
    // IFG
    interFrameGap = INTERFRAME_GAP_BITS / (double) FAST_ETHERNET_TXRATE;
    // the safety interval for testing that the frame fits in this slot, use the IFG
    // this value should be at least 4 * IFG and bigger as the switch relayunit processing time
    safetyInterval = par("safetyIntervalLength");
    safetyInterval += interFrameGap; // include additional time for the IFG of the MACs
    WATCH(safetyInterval);
    missingSlots.setName("missed slots");
}

void MACRelayUnitNPTDMA::handleMessage(cMessage *msg)
{
    if (msg == slotTimeEvent) {
        // advance the time slot, the slot counting starts with zero
        if (actualSlot < noRealTimeSlot) {
            //EV << "change time slot from " << actualSlot << " to " << actualSlot + 1 << endl;
            actualSlot++;
        }
        else {
            //EV << "change time slot from " << actualSlot << " to 0" << endl;
            actualSlot = 0;
        }
        // schedule next time slot
        scheduleAt(simTime() + timeSlotLength, slotTimeEvent);
    }
    else
    {
        // send it to the base class
        MACRelayUnitNP::handleMessage(msg);
    }
}

void MACRelayUnitNPTDMA::handleAndDispatchFrame(EtherFrame *frame, int inputport)
{
    // update address table
    updateTableWithAddress(frame->getSrc(), inputport);

    // handle broadcast frames first
    if (frame->getDest().isBroadcast())
    {
        EV << "Broadcasting broadcast frame " << frame << endl;
        broadcastFrame(frame, inputport);
        return;
    }

    simtime_t time = simTime();
    int outputport = -1;

    // Try to finds output port from real-time schedule
    if (schedulePresent)
        outputport = getPortFromSchedule(inputport);

    // if not found try to find output port of destination address
    if (outputport == -1)
        outputport = getPortForAddress(frame->getDest());

    // sends to output port, if not found then broadcasts to all other ports instead
    if (inputport == outputport)
    {
        EV << "Output port is same as input port, " << frame->fullName() <<
              " dest " << frame->getDest() << ", discarding frame" << endl;
        delete frame;
        return;
    }
    if (outputport >= 0) // we know the port
    {
        simtime_t delay, slotBoundary;
        delay = 0;
        slotBoundary = calcSlotBoundary();

        // TODO make this logic more modular

        /* we have five cases:
          * - we handle a real-time frame, and the frame fit            : delay is zero
          * - we handle a real-time frame, and the frame don't fit    : delay is timeslot count
          * - we handle a no real-time frame, the frame fit and the 
          *   output port is free for no-real-time traffic                    : delay is zero
          * -  .. the frame don't fit, or the port is not free
          *   and we are at the no-real-time port                            : delay is the next free slot plus one
          *   or we are at a real-time slot                                       : delay is the next free slot minus actual slot 
          */

        if (isInputPortRealTime(inputport) && actualSlot != noRealTimeSlot) { // and we handle a real-time frame
            // if the frame fits in the current slot
            if((time - slotBoundary + safetyInterval) < timeSlotLength) {
                send(frame, "lowerLayerOut", outputport);
                EV << "Sending real-time frame " << frame << " with dest address " << frame->getDest() << " at the current slot to port " << outputport << endl;
            }
            else {
                // schedule in the next cycle
                delay = slotBoundary;
                delay += timeSlotCount * timeSlotLength;
                EV << "Sending real-time frame " << frame << " with dest address " << frame->getDest() << " at the slot " << actualSlot << " at " << delay << "s " <<  " in the next cycle to port " << outputport << endl;
                delay = delay - time;
                sendDelayed(frame, delay, "lowerLayerOut", outputport);
            }
        }
        else { // handle a no real-time frame
            // if the frame fits in the current slot and the output port is free for no-real-time traffic
            // no need to check for the no-real-time slot
            if((time - slotBoundary + safetyInterval) < timeSlotLength && !isOutputPortRealTime(outputport)) {
                send(frame, "lowerLayerOut", outputport);
                EV << "Sending frame " << frame << " with dest address " << frame->getDest() << " at the current slot to port " << outputport << endl;
            }
            else {
                // schedule in the next free slot, or in the next cycle
                delay = slotBoundary;
                unsigned int targetSlot;

                // special handling of the no-real-time slot
                if (actualSlot == noRealTimeSlot) {
                    targetSlot = noRealTimeSlot;
                    // find the next free slot, or use the NRT slot
                   for (; targetSlot < noRealTimeSlot; targetSlot++) {
                       if (! isOutputPortRealTime(outputport, targetSlot))
                           break;
                   }
                    delay += (targetSlot + 1) * timeSlotLength;
                    if (targetSlot == noRealTimeSlot)
                        EV << "Sending frame " << frame << " with dest address " << frame->getDest() << " at the slot " << targetSlot << " at " << delay << "s " <<  " in this cycle to port " << outputport << endl; 
                    else
                        EV << "Sending frame " << frame << " with dest address " << frame->getDest() << " at the slot " << targetSlot << " at " << delay << "s " <<  " in the next cycle to port " << outputport << endl;
                }
                else {
                    targetSlot = actualSlot + 1;
                    // find the next free slot, or use the NRT slot
                    for (; targetSlot < noRealTimeSlot; targetSlot++) {
                       if (! isOutputPortRealTime(outputport, targetSlot))
                           break;
                   }
                    delay += (targetSlot - actualSlot) * timeSlotLength;
                    EV << "Sending frame " << frame << " with dest address " << frame->getDest() << " at the slot " << targetSlot << " at " << delay << "s " <<  " in this cycle to port " << outputport << endl;
                }
                delay = delay - time;
                sendDelayed(frame, delay, "lowerLayerOut", outputport);
            }
        }
        // plausability check
        if (delay > (2 * timeSlotCount * timeSlotLength)) {
            error("Schedule message to much to the future, this is presumable a error");
        }
    }
    else
    {
        EV << "Dest address " << frame->getDest() << " unknown, broadcasting frame " << frame << endl; 
        broadcastFrame(frame, inputport);
    }
}

int MACRelayUnitNPTDMA::getPortFromSchedule(const unsigned int inputPort)
{
    // not found
    int tmp = -1;

    vector<IOHolder> vec = (*schedule)[actualSlot];
    vector<IOHolder>::iterator it = vec.begin();
    while (it != vec.end()) {
        if ((*it).getInputPort() == inputPort) {
            tmp = (*it).getOutputPort();
            break;
        }
        it++;
    }
    // no schedule found
    if (tmp == -1) {
        // check if the input port has any RT traffic assigned
        //bool RTstate = false;
        //for (unsigned int i = 0; i < timeSlotCount; i++) {
        //    if (isInputPortRealTime(inputPort, i)) {
        //        RTstate = true;
        //        break;
        //    }
        //}

        // cout << "no schedule found for port " << inputPort << endl;
        // cout << "current slot is " << actualSlot << " at " << simTime() << " s" << "\n";
        // report and count errors in RT-slots 
        if (actualSlot != noRealTimeSlot) {
            cout << "missed real-time slot at " << simTime() << "s" << endl;
            cout << "current slot is " << actualSlot << "\n";
            cout << "input port is " << inputPort << "\n"; 
            EV << "Missed real-time slot, current slot is " << actualSlot << endl;
            missedSlots++;
            missingSlots.record(missedSlots);
        }
        //else // no real-time port, this must be a NRT frame
        //    cout << "received no real-time frame" << "\n";
    }
    //else 
    //    cout << "real-time outputport: " << tmp << "\n";
    return tmp;
}

void MACRelayUnitNPTDMA::broadcastFrame(EtherFrame *frame, const int inputport)
{
    simtime_t delay, slotBoundary, time;
    time = simTime();
    delay = slotBoundary = 0;
    slotBoundary = calcSlotBoundary();

    cout << "Broadcast frame at sim time " << simTime() << endl;

    // TODO perhaps use a similar logic like the MAC

    /* 
      * we have three cases:
      * - we are already in the right slot and the frame fit          : delay is zero
      * - we are already in the right slot, and the frame don't fit : delay is timeSlotCount
      * - we need to wait for the right slot                                : delay is target slot minus actual slot
    */

    // we are in the right slot
    if (actualSlot == noRealTimeSlot) {
        // if the current frame fits in this slot, schedule in this slot
        if((time - slotBoundary + safetyInterval) < timeSlotLength)
            EV << "Broadcast at current slot" << endl;
        else {
            // schedule in the next cycle
            delay = slotBoundary;
            delay += timeSlotCount * timeSlotLength;
            EV << "Broadcast at slot " << noRealTimeSlot << " in the next cycle at " << delay << "s" << endl;
            delay = delay - time;
        }
    }
    else {
        // we need to wait for the no real-time slot
        delay = slotBoundary;
        delay += (noRealTimeSlot - actualSlot) * timeSlotLength;
        EV << "Broadcast at slot " << noRealTimeSlot << " at " << delay << "s" << endl;
        delay = delay - time;
    }
    // plausability check
    if (delay > (2 * timeSlotCount * timeSlotLength)) {
        error("Schedule broadcast message to much to the future , this is presumable a error");
    }

    for (int i = 0; i < numPorts; ++i)
        if (i!=inputport)
            sendDelayed((EtherFrame*)frame->dup(), delay, "lowerLayerOut", i);
    delete frame;
}

void MACRelayUnitNPTDMA::finish()
{
    if (par("writeScalars").boolValue()) {
        recordScalar("missed real-time slots", missedSlots);
    }
    MACRelayUnitNP::finish();
}

simtime_t MACRelayUnitNPTDMA::calcSlotBoundary() const
{
    // calculate offset to no-rt slot, first the slot boundary, truncate the value
    return timeSlotLength * trunc(simTime() / timeSlotLength);
}

bool MACRelayUnitNPTDMA::isInputPortRealTime(const unsigned int port) const
{
    return isInputPortRealTime(port, actualSlot);
}

bool MACRelayUnitNPTDMA::isInputPortRealTime(const unsigned int port, const unsigned int slot) const
{
    bool tmp = false;
    // check if the port has real-time traffic in this slot 
    vector<IOHolder> vec = (*schedule)[slot];
    vector<IOHolder>::iterator it = vec.begin();
    while (it != vec.end()) {
        if ((*it).getInputPort() == port) {
            tmp = true;
            break;
        }
        it++;
    }
    return tmp;
}

bool MACRelayUnitNPTDMA::isOutputPortRealTime(const unsigned int port) const
{
    return isOutputPortRealTime(port, actualSlot);
}

bool MACRelayUnitNPTDMA::isOutputPortRealTime(const unsigned int port, const unsigned int slot) const
{
    bool tmp = false;
    // check if the port has real-time traffic in this slot 
    vector<IOHolder> vec = (*schedule)[slot];
    vector<IOHolder>::iterator it = vec.begin();
    while (it != vec.end()) {
        if ((*it).getOutputPort() == port) {
            tmp = true;
            break;
        }
        it++;
    }
    return tmp;
}
