/*
 * Copyright (C) 2006 Henning Westerholt
 * Copyright (C) 2003 CTIE, Monash University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifdef _MSC_VER
#pragma warning(disable:4786)
#endif

#include <stdlib.h>
#include <stdio.h>
#include <cxmlelement.h>

#include <omnetpp.h>
#include "EtherMACTDMA.h"

using namespace std;

Define_Module( EtherMACTDMA );

EtherMACTDMA::EtherMACTDMA() : timeSlotCount(0), timeSlotLength(0), actualSlot(0), slotCounter(0), slotId(0), isSlotted(false), safetySlot(0), safetyInterval(0), testEqual(0), interFrameGap(0) {}

void EtherMACTDMA::initialize() 
{
    // call the base class
    EtherMAC::initialize();

    timeSlotLength = par("timeSlotLength");
    isSlotted = timeSlotLength > 0;
    WATCH(timeSlotLength);
    WATCH(isSlotted);

    if (!isSlotted)
        EV << "Deactivate TDMA scheduling" << endl;
    else {
        // open the schedule file
        cXMLElement * scheduleConfig;
        scheduleConfig = par("scheduleConfigFile");
        // get the timeslot count
        timeSlotCount = atoi(scheduleConfig->getAttribute("timeSlotCount"));
        // timeSlotCount starts with 0, and append a slot for no-realtime traffic
        // TODO 64 bytes is not enough for no-rt
        timeSlotCount += 2;

        // construct a parameter resolver
        cXMLElement::ParamResolver * resolver = new ModNameParamResolver(this);
        // get the xml sub tree for this device, this is where the swith id equals the parent module name
        scheduleConfig = scheduleConfig->getElementByPath("//device[@id=$PARENTMODULE_FULLNAME]", scheduleConfig, resolver);

        if (scheduleConfig != 0)
            slotId = atoi(scheduleConfig->getAttribute("slotId"));
        else {
            slotId = timeSlotCount; // we are not a real-time device
            EV << "warning, MAC is slotted, but has no assigned slot id" << endl;
        }
        delete resolver;
        // schedule first time slot event
        slotTimeEvent = new cMessage("slot");
        scheduleAt(simTime() + timeSlotLength, slotTimeEvent);
    }

    if (slotId < 0)
        error("MAC has invalid slot id %s set: use a positive id or null", slotId);
    WATCH(slotId);
    WATCH(timeSlotCount);
    WATCH(actualSlot);

    // initalize variables for the scheduling
    // add a small amount of time for safe matching of the timeslot
    // this value could be zero if the switch processing time is > 0s
    safetySlot = timeSlotLength / (double) 1000;

    // IFG
    interFrameGap = INTERFRAME_GAP_BITS / (double) FAST_ETHERNET_TXRATE;

    // the safety interval for testing that the frame fits in this slot
    // this value should be at least the switch relayunit processing time, or if that value is set to zero, the IFG
    safetyInterval = par("safetyIntervalLength");
    safetyInterval += interFrameGap; // include additional time for the IFG of the MACs
    WATCH(safetyInterval);

    // check if two times are equal, the normal range is 10^-5 - 10^-6, so this should be safe
    testEqual  = 1 / (double) 1000000000;
}

// overwrite the different methods
void EtherMACTDMA::scheduleEndIFGPeriod()
{
    if (isSlotted)
    {
//         cout << "actual slot " << actualSlot << " and time " << simTime() << "\n";
        /* 
         * calculate the sending time:
         * the actual time is the base, align this time to a slot boundary
         * and calculate the offset to the wanted slot
         */
        simtime_t delay, time, slotBoundary;
        delay = time = simTime();
        // calculate the slot boundary, round to zero (truncate)
        slotBoundary = timeSlotLength * trunc((time / timeSlotLength));

        // this is not zero if the slot from the time differs from the actual slot 
        int pendingSlotChange = checkSlotBoundary(slotBoundary);

        /* 
           To enforce the TDMA, we wait longer then the standard IFG
           this is allowed according the standard.
           Quote from IEEE 802.3 - 4.2.3.2.2:
           Note that interFrameSpacing is the minimum value of the interframe
           spacing. If necessary for implementation reasons, a transmitting 
           sublayer may use a larger value with a resulting decrease in its 
           throughput.
        */

        /* 
         * we have four cases:
         * - we are already in the right slot and the frame fit        : delay is zero
         * - we are already in the right slot, and the frame don't fit : delay is timeSlotCount
         * - we need to wait for the right slot in this cycle          : delay is target slot minus actual slot
         * - we need to wait for the right slot in the next cycle      : delay is timeSlotCount minus actual slot plus target slot
         * 
         * Handle pending chances safely, if there is a pending time slot chance, thus the slotBoundary time is one slot advanced from the actual slot:
         * - we are not at the right slot, wait for the next cycle
         * - we need to add one to the actual slot the other cases
        */

        // if we are in the right real-time slot
        if (actualSlot == slotId) {
            // and if the current frame fits in this slot plus IFG plus additional safety factor for switch processing and cable delay and we are not at a slot boundary
            if ((time - slotBoundary + safetyInterval) < timeSlotLength && pendingSlotChange == 0) {
                // wait for the IFG, we want compatibility to normal ethernet
                delay = time + interFrameGap;
                //cout << "Schedule at the current slot at " << time << endl;
                EV << "Schedule at the current slot" << endl;
            }
            else {
                // we need to wait for the next cycle
                delay = slotBoundary;
                switch (pendingSlotChange) {
                    case 0  : delay += (timeSlotCount * timeSlotLength); break;
                    case 1  :  delay += (timeSlotCount - 1)* timeSlotLength; break;
                    case -1 : delay += (timeSlotCount + 1) * timeSlotLength; break;
                }
                delay += safetySlot;
                //cout << "Schedule at slot " << slotId << " in the next cycle at " << delay << "s" << endl;
                EV << "Schedule at slot " << slotId << " in the next cycle at " << delay << "s" << endl;
             }
        }
        else { // wrong slot
            delay = slotBoundary;

            if (actualSlot < slotId) { // we need to wait for our time slot
                switch (pendingSlotChange) {
                    case 0  : delay += ((slotId - actualSlot) * timeSlotLength); break;
                    case 1  : delay += ((slotId - actualSlot - 1) * timeSlotLength); break;
                    case -1 : delay += ((slotId - actualSlot + 1) * timeSlotLength);
                }
                delay += safetySlot;
                EV << "Schedule at slot " << slotId << " at " << delay << "s" << endl;
                //cout << "Schedule at slot " << slotId << " at " << delay << "s" << endl;
            }
            else { // actualSlot > slotId, we need to wait for the next cycle
                    switch (pendingSlotChange) {
                        case 0  : delay += ((timeSlotCount - actualSlot + slotId) * timeSlotLength); break;
                        case 1  : delay += ((timeSlotCount - actualSlot + slotId - 1) * timeSlotLength); break;
                        case -1 : delay += ((timeSlotCount - actualSlot + slotId + 1) * timeSlotLength); break;
                 }
                delay += safetySlot;
                EV << "Schedule at slot " << slotId << " in the next cycle at " << delay << "s" << endl;
                //cout << "Schedule at slot " << slotId << " in the next cycle at " << delay << "s" << endl;
            }

        }
        // plausability check
        if (delay - time > (2 * timeSlotCount * timeSlotLength)) {
             error("Schedule message to much to the future, this is presumable a error");
        }

        scheduleAt(delay, endIFGMsg);
        transmitState = WAIT_IFG_STATE;
        }
    else
        EtherMAC::scheduleEndIFGPeriod();
}

void EtherMACTDMA::handleMessage(cMessage *msg)
{
    if (msg == slotTimeEvent) {
        // advance the time slot, the slot counting starts with zero
        if (actualSlot < timeSlotCount - 1) {
            //EV << "change time slot from " << actualSlot << " to " << actualSlot + 1 << endl;
            actualSlot++;
        }
        else {
            //EV << "change time slot from " << actualSlot << " to 0" << endl;
            actualSlot = 0;
        }
        // schedule next time slot
        scheduleAt(simTime() + timeSlotLength, slotTimeEvent);
        // advance the time slot counter
        slotCounter++;
    }
    else
    {
        // send it to the base class
        EtherMAC::handleMessage(msg);
    }
}

// return the offset to the actual slot from the time based slot
int EtherMACTDMA::checkSlotBoundary(const simtime_t slotBoundary) const
{
    simtime_t time = simTime();
    int tmp = 0; // not at boundary
    /*
    * we are at a slot boundary if the simTime is equal the slotBoundary time or
    * the fabs(time - slotBoundary - timeSlotLength) < testEqual and
    * if the slot number calculated from the time is equal the actual slot number
    */
    if (fabs(time - slotBoundary) < testEqual || fabs(time - slotBoundary - timeSlotLength) < testEqual) {
        unsigned int actualSlotFromTime = (unsigned int) trunc(time / timeSlotLength);

        if (actualSlotFromTime != slotCounter) {
            if (actualSlotFromTime > slotCounter)
                tmp = 1;
            if (actualSlotFromTime < slotCounter)
                tmp = -1;
            //cout << "slot from time: " << actualSlotFromTime << " slot counter " << slotCounter << endl;
            //cout << "pending time slot chance at time " << time << " s" << endl;
        }
    }
    return tmp;
}
