#!/bin/bash
#
# Copyright (C) 2006 Henning Westerholt
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
#
# convertNetwork:
#
# Bash script to convert a network from the "PG"-format to the INET/ OMNeT++ format.
# Needs xlstproc from the libxslt package, sed, grep and gnu-coreutils.
# Usage: ./convertNetwork [NETWORK] [COMMUNICATIONLINEs] [SCHEDULE]
#
# Next time do this stuff in python.. ;-)

echo "Convert a network description from the PG-format in the OMNeT++ format."
echo

# input checking and setup

if [ $# -lt 3 ]
# Exit if to few arguments given.
then
  echo "Usage: $0 [NETWORK] [COMMUNIKATIONLINES] [SCHEDULE]"
  echo "The input must be valid XML files in the 'PG'-Format."
exit 1
fi

# we have three input arguments

NET=`basename $1 .xml`-out-network.xml
TRAFFIC=`basename $1 .xml `-out-traffic.ini
SCHEDULE=`basename $1 .xml`-out-schedule.xml


# do the work

# convert the network definition
echo "convert the network.."
xsltproc -o $NET network.xsl $1 || exit 1

# convert the traffic definition
echo "convert the traffic.."
xsltproc -o $TRAFFIC traffic.xsl $1 || exit 1

# create a random temp file
TMPFILE=`mktemp -p . tmp-xml.XXXXXXXXXX` || exit 1

# catch signals and cleanup
trap 'test -e $TMPFILE && rm $TMPFILE; exit 0' 2 15

# concat the communicationslines, the schedules and the network
# add a pseudo xml top element
echo "<temp-file>" > $TMPFILE
cat $1 $2 $3 >> $TMPFILE
echo "</temp-file>" >> $TMPFILE

# convert the schedules
echo "convert the schedules.."
xsltproc -o "$SCHEDULE" schedule.xsl $TMPFILE || exit 1


# post-processing and cleanup
# remove unnessecary whitespaces
cp $TRAFFIC $TMPFILE
sed -e '/^ *$/d' $TMPFILE > $TRAFFIC
cp $SCHEDULE $TMPFILE
sed -e '/^ *$/d' $TMPFILE > $SCHEDULE

# convert xml network to OMNeT++ ned language
echo "convert the network to ned format for OMNeT++.."
nedtool -n -y $NET

# calculate the maximum timeslot value
# get the timeslot node from the xml
grep "timeSlot id" $SCHEDULE  > $TMPFILE
# remove trailing whitespace and the timeslot node xml code. Sort the remaining, and get the first.
MAX=`sed -e 's/^[ \t]*//' -e 's/<timeSlot id="//' -e 's/".*>//' $TMPFILE | sort -n | tail -n 1`
echo "The number of time slots in the schedule is $MAX."

cp $SCHEDULE $TMPFILE
# insert this value in the schedule xml
sed -e "s/schedule timeSlotCount=\"\"/schedule timeSlotCount=\"$MAX\"/g" $TMPFILE > $SCHEDULE

rm $TMPFILE


# finished
echo ""
echo "Finish network converting, the generated files are:"
echo "$NET, $TRAFFIC, $SCHEDULE."

# TODO do error checking of xslproc
exit 0
