//
// Copyright (C) 2006 Henning Westerholt
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//

#include <iostream>
#include <math.h>
#include <QString>

#include <network.h>
#include <hardware.h>
#include <ports.h>
#include <utility.h>
#include <links.h>
#include <messages.h>

using namespace std;

NetworkParameter::NetworkParameter(): networkSize(0), siblingChance(0), connectionChance(0), connectionRange(0), normalDeviceChance(0), randomSeed(0), trafficMode(Invalid), valid(false) {}

Network::Network(const NetworkParameter& parameter) : parameter(parameter), myTree(), myLinks(), myMessages()

{
    utility = Utility::getInstance(parameter.randomSeed);
}

Network::~Network()
{
    delete myTree;
    // @todo delete the content too
    delete myLinks;
    delete myMessages;
}

// convinience method
void Network::createNetwork()
{
    createTree();
    decorateTree();
    connectTree();
}

// create a hardware tree
void Network::createTree()
{
    // create a tree with a root node
    myTree = new tree<Hardware>(createHardware());
    tree<Hardware>::iterator root, tmp;
    // assign the root to the first node
    root = myTree->begin();

    // create a network of size "networkSize"
    //start with count = 1 for the root node
    for (unsigned int count = 1; count < parameter.networkSize; count++) {
        // we want at least two siblings, for a bus topology
        tmp = myTree->append_child(root, createHardware());
        tmp = myTree->append_child(root, createHardware());
        count++;

        // the chance for more siblings is about siblingChance
        while (utility->getDoubleRandomNumber() < parameter.siblingChance && count < parameter.networkSize)  {
            tmp = myTree->append_child(root, createHardware());
            count++;
         }
         // the new root is the last inserted child
         // TODO test if it make sense to choose a random node
         root = tmp;
    }
}

// set the correct hardware type for the tree nodes
void Network::decorateTree()
{
    tree<Hardware>::iterator iter = myTree->begin();

    // the root has no uplink port
    bool isRoot = true;

    // assign hardware types to the tree
    //@todo support more hardware types
    while (iter != myTree->end())  {

        // if the node don't have more childen, then it's
        // of type netdevice, otherwise it's a switch
        if (iter.number_of_children() == 0)
        {
            // choose a device type
            if (utility->getDoubleRandomNumber() < parameter.normalDeviceChance)
                (*iter).setType("NetDevice");
            else
                (*iter).setType("RTDevice");
            (*iter).appendPort(createPort());
        }
        else {
            // if it's a switch, we have more than one port
            (*iter).setType("RTSwitch");
            // append one port for every child, and we need also a port as uplink
            unsigned int i;
            // the root has no uplink port
            if (!isRoot)
                i = 0;
            else {
                i = 1;
                isRoot = false;
            }

            for (; i <= iter.number_of_children(); i++) {
                Port tmp = createPort();
                tmp.setPortNr(i);
                (*iter).appendPort(tmp);
            }
        }
        iter++;
    }
    iter = myTree->begin();

}

// create explicit links and messages
void Network::connectTree()
{
    tree<Hardware>::iterator iter = myTree->begin();
    // fill in the the uploadPortNumber and create links

    // the root get uploadPortNumber - 1
    (*iter).setUploadPortNumber(-1);
    // every device is connected through port 0 to the parent
    // and for now, we assume the same for all switches

    // the root has no uplink port
    bool isRoot = true;

    // create the links
    myLinks = new Links();

    while (iter != myTree->end()) {
        tree<Hardware>::sibling_iterator sibIter = myTree->begin(iter);
        unsigned int portCnt;

        if (!isRoot)
            portCnt = 1; // start with port 1, port 0 is the uplink
        else {
            portCnt = 0; // the root has no uplink
            isRoot = false;
        }

        // create the links subelements
        while (sibIter != myTree->end(iter)) {
            Link tmp = createLink();
            tmp.setFromDevice((*iter).getId());
            // count the ports
            tmp.setFromPort(portCnt++);
            tmp.setToDevice((*sibIter).getId());
            myLinks->append(tmp);

            sibIter++;
        }
        iter++;
    }

    //create the messages
    myMessages = new Messages();
    tree<Hardware>::iterator recIter;
    // reset the id counter
    utility->resetUniqueNumber();
    unsigned int networkSize = myTree->size();

    switch (parameter.trafficMode) {

        // create random messages
        case Random: {
            recIter = iter = myTree->begin();
            // walk over the tree
            while (iter != myTree->end()) {

                // the chance for a connection of a node is about connectionChance
                // @todo don't assign more than one message to two partners
                // @todo this don't stops for connectionChance = 100%
                while (utility->getDoubleRandomNumber() < parameter.connectionChance) {

                    // switches don't send messages
                    if (iter.number_of_children() == 0) {
                        // switches don't receive messages
                        while (recIter.number_of_children() != 0) {
                            // start from the beginning
                            recIter = myTree->begin();
                            // advance the receiver iterator about network size and 
                            // connectionRange factor and a random component, 
                            // use "ceil" to avoid the null
                            recIter+=static_cast<unsigned int>(ceil(utility->getDoubleRandomNumber() 
                                * (parameter.connectionRange * parameter.networkSize)));
                        }
                        // don't make a connection to self
                        if (iter != recIter) {
                            // make connections
                            Message tmp = createMessage();
                            tmp.setSenderId((*iter).getId());
                            tmp.setReceiverId((*recIter).getId());
                            myMessages->append(tmp);
                        }
                        // set the receiver iterator back
                        recIter = myTree->begin();
                    }
                }
                iter++;
            }
            break;
        }
        // one master send all messages, multi-master is a modification of master
        case Master: case MultiMaster: {
            recIter = iter = myTree->begin();
            // choose a random master from the network
            iter+=static_cast<unsigned int>(ceil(utility->getDoubleRandomNumber() * networkSize));
            // assure that the master is not a switch
            // @todo this method don't work all the time, obviously
            while (iter.number_of_children() != 0) {
                iter++;
             }
            // the network size is a upper bound for the messages
            for (unsigned int i = 0; i < networkSize; i++) {
                // the chance for a connection of a node is about connectionChance
                // @todo don't assign more than one message to two partners
                while (utility->getDoubleRandomNumber() < parameter.connectionChance) {
                    // switches don't receive messages
                    while (recIter.number_of_children() != 0) {
                        // start from the beginning
                        recIter = myTree->begin();
                        // advance the receiver iterator about network size and 
                        // connectionRange factor and a random component, 
                        // use "ceil" to avoid the null
                        recIter+=static_cast<unsigned int>(ceil(utility->getDoubleRandomNumber() 
                            * (parameter.connectionRange * parameter.networkSize)));
                    }
                    // don't make a connection to self
                    if (iter != recIter) {
                        // make connections
                        Message tmp = createMessage();
                        tmp.setSenderId((*iter).getId());
                        tmp.setReceiverId((*recIter).getId());
                        myMessages->append(tmp);
                    }
                    // set the receiver iterator back
                    recIter = myTree->begin();
                }
                // advance the master iterator with a 3% chance in multi-master operation
                // this is a good value, perhaps make this changeable too
                if (parameter.trafficMode == MultiMaster && utility->getDoubleRandomNumber() < 0.03)
                    iter++;
                    // assure that the new master is not a switch
                    // @todo this method don't work all the time, obviously
                    while (iter.number_of_children() != 0) {
                        iter++;
                    }
                }
            break;
        }
        default: break; // this should not happen
    }
}

const tree<Hardware> * Network::getTree() const
{
    return myTree;
}

const Links * Network::getLinks() const
{
    return myLinks;
}

const Messages * Network::getMessages() const
{
    return myMessages;
}

// debugging method
void Network::printTree() const
{
    tree<Hardware>::iterator iter = myTree->begin();
    while (iter != myTree->end())  {
        for (int i = 0; i < myTree->depth(iter); i++)
            cout << " " ;
        cout << QString::number((*iter).getId()).toStdString() << endl;
        iter++;
    }
}

// helper methods
Hardware Network::createHardware() const
{
    // type depends on the context, will be set later
    Hardware hw;
    hw.setId(utility->getUniqueNumber());
    hw.setMode("f");
    hw.setBandwith(100000000 * 8); // 100 MBit/s in bits/sec
    hw.setMac(utility->generateAutoMACAddress());
    return hw;
}

Port Network::createPort() const
{
    Port port;
    port.setCableId(- utility->getUniqueNumber());
    port.setSpeed(100000000 * 8); // 100 MBit/s in bits/sec
    // portNr depends on the hardware type, will be set later
    return port;
}

Link Network::createLink() const
{
    Link link;
    // every device is connected through port 0 to the parent
    // and for now, we assume the same for all switches
    link.setToPort(0);
    // delay is zero for now
    link.setDelay(0);
    return link;
}

Message Network::createMessage() const
{
    Message msg;
    msg.setMsgId(utility->getUniqueNumber());
    return msg;
}
