//
// Copyright (C) 2006 Henning Westerholt
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//

#include <QMutex>
#include <QDate>
#include <QTime>
#include <utility.h>
/* 
 * use the mersenne twister pseudo random generator, this is a fast, 
 * equidistributed generator with a very long period, the 
 * "'random number generator of choice' for statistical simulations[..]" 
 * (Wikipedia)
 */
#include <mersenneTwister.h>

Utility::Utility(unsigned int randomSeed) : counter(0), macCounter(0), randomSeed(randomSeed)
{
    if (randomSeed == 0) {
        // setup of the random generator with the current time and date
        QTime time = QTime::currentTime();
        QDate date = QDate::currentDate();
        randomSeed = ((time.msec() * time.second()) * time.minute() * date.day()) / date.day();
    }
    init_genrand(randomSeed);
    macCounter = genrand_int31(); // so we have at least 2^31 spare mac addresses
}

Utility::~ Utility()
{
    delete instance;
}

// initialize static class variable
Utility * Utility::instance = NULL;

// Singleton, adapted from http://www.tekpool.com/?p=35
Utility * Utility::getInstance(const unsigned int randomSeed)
{
    QMutex mutex;
    if (instance == NULL) // first check
    {
        mutex.lock();
        if (instance == NULL) // second check
        {
            instance = new Utility(randomSeed);
        }
        mutex.unlock();
    }
    return instance;
}

/*
 * modified copy from INET framework, file
 * ./NetworkInterfaces/Contract/MACAddress.cc
 * author: Monash University, under GPL Licence
 */
QString Utility::generateAutoMACAddress()
{
    unsigned char addrbytes[6];

    addrbytes[0] = 0x0A;
    addrbytes[1] = 0xAA;
    addrbytes[2] = (macCounter >> 24) & 0xff;
    addrbytes[3] = (macCounter >> 16) & 0xff;
    addrbytes[4] = (macCounter >> 8) & 0xff;
    addrbytes[5] = (macCounter) & 0xff;
    macCounter++;

    QString str;
    for (int i = 0; i < 6; i++)
    {
        // append as hex string
        str.append(QString::number(addrbytes[i], 16));
        /*
        if (i < 5)
            str.append(":");
        */
    }

    str = str.toUpper();
    return str;
}

unsigned long Utility::getUniqueNumber()
{
    return counter++;
}

unsigned long Utility::getRandomNumber() const
{
    return genrand_int32();
}

double Utility::getDoubleRandomNumber() const
{
    return genrand_real1();
}

void Utility::resetUniqueNumber()
{
    counter = 0;
}

bool Utility::isPercent(const double val)
{
    return (val >= 0 && val <= 1);
}
